import 'package:flutter/material.dart';
import 'package:easy_localization/easy_localization.dart';
import 'package:feather_icons/feather_icons.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:wordpress_app/blocs/config_bloc.dart';
import 'package:wordpress_app/cards/card4.dart';
import 'package:wordpress_app/config/config.dart';
import 'package:wordpress_app/models/article.dart';
import 'package:wordpress_app/services/wordpress_service.dart';
import 'package:wordpress_app/utils/empty_image.dart';
import 'package:wordpress_app/utils/loading_card.dart';
import 'package:wordpress_app/services/app_service.dart';
import 'package:wordpress_app/widgets/inline_ads.dart';
import 'package:wordpress_app/widgets/loading_indicator_widget.dart';
import 'package:wordpress_app/widgets/scroll_to_top_button.dart';

class VideoTab extends ConsumerStatefulWidget {
  const VideoTab({super.key});

  @override
  ConsumerState<VideoTab> createState() => _VideoTabState();
}

class _VideoTabState extends ConsumerState<VideoTab>
    with AutomaticKeepAliveClientMixin {
  final List<Article> _articles = [];
  ScrollController? _controller;
  int _page = 1;
  bool? _loading;
  bool? _hasData;
  var scaffoldKey = GlobalKey<ScaffoldState>();
  final int _postAmount = 10;

  @override
  void initState() {
    _controller =
        ScrollController(initialScrollOffset: 0.0, keepScrollOffset: true);
    _controller!.addListener(_scrollListener);
    _fetchArticles();
    _hasData = true;
    super.initState();
  }

  @override
  void dispose() {
    super.dispose();
    _controller!.dispose();
  }

  Future _fetchArticles() async {
    await WordPressService().fetchVideoPosts(_page, _postAmount).then((value) {
      _articles.addAll(value);
      if (_articles.isEmpty) {
        _hasData = false;
      }
      setState(() {});
    });
  }

  Future<void> _scrollListener() async {
    var isEnd = AppService.isScrollNearEnd(_controller!);
    if (isEnd && _articles.isNotEmpty) {
      setState(() {
        _page += 1;
        _loading = true;
      });
      await _fetchArticles().then((value) {
        setState(() {
          _loading = false;
        });
      });
    }
  }

  Future _onRefresh() async {
    setState(() {
      _loading = null;
      _articles.clear();
      _hasData = true;
      _page = 1;
    });
    await _fetchArticles();
  }

  @override
  Widget build(BuildContext context) {
    final configs = ref.read(configBlocProvider).configs!;
    super.build(context);
    return Scaffold(
      key: scaffoldKey,
      appBar: AppBar(
        title: const Text('video-contents').tr(),
        titleTextStyle: Theme.of(context).textTheme.titleLarge?.copyWith(
              fontWeight: FontWeight.w600,
            ),
        actions: [
          IconButton(
            icon: const Icon(
              FeatherIcons.rotateCw,
              size: 22,
            ),
            onPressed: () async => await _onRefresh(),
          )
        ],
      ),
      body: RefreshIndicator(
        color: Colors.white,
        backgroundColor: Theme.of(context).primaryColor,
        onRefresh: () async => await _onRefresh(),
        child: Stack(
          children: [
            CustomScrollView(
              controller: _controller,
              slivers: [
                if (_hasData == false)
                  SliverFillRemaining(
                    child: Container(
                      alignment: Alignment.center,
                      child: EmptyPageWithImage(
                          image: Config.noContentImage,
                          title: 'no-contents'.tr()),
                    ),
                  )
                else
                  SliverPadding(
                    padding: const EdgeInsets.all(20),
                    sliver: SliverList.separated(
                      itemCount:
                          _articles.isEmpty ? 6 : _articles.length,
                      separatorBuilder: (context, index) =>
                          const Divider(height: 40),
                      itemBuilder: (context, index) {
                        if (_articles.isEmpty && _hasData == true) {
                          return const LoadingCard(height: 280);
                        } else if (index < _articles.length) {
                          if ((index + 1) % configs.postIntervalCount == 0) {
                            return Column(
                              children: [
                                Card4(
                                    article: _articles[index],
                                    heroTag: 'video${_articles[index].id}'),
                                const Divider(height: 20),
                                const InlineAds(),
                              ],
                            );
                          } else {
                            return Card4(
                                article: _articles[index],
                                heroTag: 'video1${_articles[index].id}');
                          }
                        }
                        return const SizedBox.shrink();
                      },
                    ),
                  ),
                if (_hasData == true)
                  SliverToBoxAdapter(
                    child: Opacity(
                      opacity: _loading == true ? 1.0 : 0.0,
                      child: const LoadingIndicatorWidget(),
                    ),
                  ),
              ],
            ),
            if (_controller != null)
              Positioned(
                bottom: 50,
                right: 16,
                child: ScrollToTopButton(controller: _controller!),
              ),
          ],
        ),
      ),
    );
  }

  @override
  bool get wantKeepAlive => true;
}
